<?php
/* --------------------------------------------------------------
  StyleConfigReader.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

/**
 * Class StyleConfigReader
 */
class StyleConfigReader
{
	private static $instance = null;
	private static $template;
	private static $styleName;
	private        $customStyles;
	private        $boxes;
	private        $styles;
	private        $reader;
	private        $errorMessage = '';
	
	
	/**
	 * @param string $p_template
	 * @param string $p_styleName
	 *
	 * @return \StyleConfigReader
	 */
	public static function getInstance($p_template, $p_styleName = null)
	{
		if(self::$instance === null)
		{
			self::$template  = $p_template;
			self::$styleName = $p_styleName;
			self::$instance  = new self;
		}
		
		return self::$instance;
	}
	
	
	protected function __construct()
	{
		$factory      = new \StyleEdit\Factories\ServiceFactory();
		$this->reader = $factory->createReadService(self::$template);
	}
	
	
	protected function __clone()
	{
	}
	
	
	/**
	 * Get Sass Code by group and style name
	 *
	 * @param string      $p_group
	 * @param string|null $p_styleName
	 *
	 * @return string
	 */
	public function getScss($p_group, $p_styleName = null)
	{
		$invalidVariables = [];
		$scssContent      = '';
		
		if($this->styles === null)
		{
			$this->styles = array();
			$styleConfig  = $this->_findStyleConfig($p_styleName);
			
			if($styleConfig !== null)
			{
				$styleConfigArray = $styleConfig->getJsonDataArray();
				
				foreach($styleConfigArray['settings'] as $setting)
				{
					if(!empty($setting) && $setting['type'] === 'display')
					{
						foreach($setting['entries'] as $entry)
						{
							if($entry['type'] === 'title' ||
							   ($entry['type'] === 'image' &&
							    ($entry['value'] === '' || $entry['value'] === "''" || $entry['value'] === '""')
							   )
							)
							{
								continue;
							}
							
							if($entry['type'] === 'image')
							{
								$entry['value'] = str_replace("'", '', $entry['value']);
								$entry['value'] = str_replace('"', '', $entry['value']);
								$entry['value'] = '\''
								                  . (StyleThemeControl::isThemeSystemActive() ? \StyleEditConfig::THEME_IMAGES_DIR : \StyleEditConfig::IMAGES_DIR) 
								                  . $entry['value'] . '\'';
							}
							
							if(!isset($this->styles[$entry['group']]))
							{
								$this->styles[$entry['group']] = array();
							}
							
							$this->styles[$entry['group']][$entry['name']] = $entry['value'];
						}
					}
				}
			}
		}
		
		if(isset($this->styles[$p_group]))
		{
			$heldBack        = [];
			$definedNames    = [];
			$sortedVariables = $this->sortScssByHierarchy($this->styles[$p_group]);
			
			foreach($sortedVariables as $var => $value)
			{
				if(is_bool($value))
				{
					$value = $value ? 'true' : 'false';
				}
				
				if(trim((string)$value) === '')
				{
					$invalidVariables[$var] = $value;
					
					continue;
				}
				
				$value = str_replace([';', '$grey'], ['', '$gray'], $value);
				
				if(array_key_exists($var, $heldBack))
				{
					foreach($heldBack[$var] as $heldBackEntry)
					{
						$scssContent .= '$' . $heldBackEntry['var'] . ': ' . $heldBackEntry['value'] . ';' . "\n";
						$definedNames[] = $heldBackEntry['var'];
					}
					unset($heldBack[$var]);
				}
				
				if(strpos($value, '$') === 0)
				{
					$valueVariableName = substr($value, 1);
					if(!in_array($valueVariableName, $definedNames, true))
					{
						if(!is_array($heldBack[$valueVariableName]))
						{
							$heldBack[$valueVariableName] = [];
						}
						$heldBack[$valueVariableName][] = ['var' => $var, 'value' => $value];
						continue;
					}
				}
				
				$scssContent .= '$' . $var . ': ' . $value . ';' . "\n";
				$definedNames[] = $var;
			}
			
			/* output anything with non-local dependencies */
			foreach($heldBack as $variable => $heldBackVariableData)
			{
				foreach($heldBackVariableData as $heldBackEntry)
				{
					$scssContent .= '$' . $heldBackEntry['var'] . ': ' . $heldBackEntry['value'] . ';' . "\n";
				}
			}
		}
		
		if(count($invalidVariables) > 0)
		{
			$message   = 'During the creation of CSS invalid empty values for the following variables were detected: %s Please enter a valid value in the configuration of your style.';
			$errorVars = '';
			$lang      = [];
			$filePath = __DIR__ . '/../lang/' . \StyleEditConfig::DEFAULT_LANGUAGE . '.json';
			
			if(file_exists($filePath))
			{
				$texts = JsonTransformer::decode(file_get_contents($filePath));
				
				if(array_key_exists('compile_error', $texts))
				{
					$message = $texts['compile_error'];
				}
			}
			
			$filePath = __DIR__ . '/../templates/' . self::$template . '/lang/' . \StyleEditConfig::DEFAULT_LANGUAGE . '.json';
			
			if(file_exists($filePath))
			{
				$lang = JsonTransformer::decode(file_get_contents($filePath));
			}
			
			foreach($invalidVariables as $var => $value)
			{
				if(array_key_exists($var, $lang))
				{
					$errorVars .= "$lang[$var] ($$var), ";
				}
				else
				{
					$errorVars .= "$$var, ";
				}
			}
			
			$errorVars           = substr($errorVars, 0, -2);
			$this->errorMessage .= sprintf($message, $errorVars) . ' ';
		}
		
		return $scssContent;
	}
	
	
	/**
	 * @param array $scssProperties
	 *
	 * @return array
	 */
	public function sortScssByHierarchy(array $scssProperties) {
		$result = [];
		
		$cycles = 0;
		while(count($scssProperties) > 0 && $cycles < 1)
		{
			$cycles++;
			foreach($scssProperties as $key => $value)
			{
				$addElement  = true;
				$dependencies = null;
				preg_match_all('/(?:\$)([^,;+ \/)]*)/', $value, $dependencies);
				if(is_array($dependencies) && count($dependencies[1]))
				{
					foreach($dependencies[1] as $parentName)
					{
						if(!array_key_exists($parentName, $result))
						{
							$addElement = false;
						}
					}
				}
				
				if($addElement)
				{
					$cycles       = 0;
					$result[$key] = $value;
					unset($scssProperties[$key]);
				}
			}
		}
		
		//add everything else at the end to ensure that all properties will be there
		if(count($scssProperties))
		{
			foreach($scssProperties as $key => $value)
			{
				$result[$key] = $value;
			}
		}
		
		return $result;
	}
	
	
	/**
	 * Returns the custom styles from a JSON File
	 *
	 * @param string|null $p_styleName
	 *
	 * @return string
	 */
	public function getCustomStyles($p_styleName = null)
	{
		if($this->customStyles === null)
		{
			$this->customStyles = '';
			$styleConfig        = $this->_findStyleConfig($p_styleName);
			
			if($styleConfig !== null)
			{
				$styleConfigArray   = $styleConfig->getJsonDataArray();
				$this->customStyles = $styleConfigArray['customStyles'];
			}
		}
		
		$scssContent = $this->customStyles;
		
		return $scssContent;
	}


	/**
	 * Searches for a setting value identified by its name. If no result is found, null will be returned.
	 * 
	 * @param $p_settingName
	 *
	 * @return mixed|null
	 */
	public function findSettingValueByName($p_settingName)
	{
		$styleConfigArray = $this->_findStyleConfig()->getJsonDataArray();
		
		if(isset($styleConfigArray['settings']) && is_array($styleConfigArray['settings']))
		{
			foreach($styleConfigArray['settings'] as $setting)
			{
				if(isset($setting['entries']) && is_array($setting['entries']))
				{
					foreach($setting['entries'] as $entry)
					{
						if(isset($entry['name']) && $entry['name'] === $p_settingName)
						{
							return (isset($entry['value'])) ? $entry['value'] : null;
						}
					}
				}
			}
		}
		
		return null;
	}
	
	
	/**
	 * Returns status (true = active, false = inactive) of a box identfied by its name.
	 *
	 * @param string $p_boxName
	 *
	 * @return bool
	 */
	public function isBoxActive($p_boxName)
	{
		$this->_initBoxes();
		
		$status = false;
		if(array_key_exists($p_boxName, $this->boxes))
		{
			$status = $this->boxes[$p_boxName]['isActive'];
		}
		
		return $status;
	}
	
	
	/**
	 * Returns the position of box identified by its name.
	 *
	 * @param string $p_boxName
	 *
	 * @return int
	 */
	public function getBoxPosition($p_boxName)
	{
		$this->_initBoxes();
		
		$position = 0;
		if(array_key_exists($p_boxName, $this->boxes))
		{
			$position = $this->boxes[$p_boxName]['position'];
		}
		
		return $position;
	}
	
	
	/**
	 * Adapter Method
	 *
	 * @param string $p_boxName
	 *
	 * @return bool
	 */
	public function get_status($p_boxName)
	{
		return $this->isBoxActive($p_boxName);
	}
	
	
	/**
	 * Adapter Method
	 *
	 * @param string $p_boxName
	 *
	 * @return string
	 */
	public function get_position($p_boxName)
	{
		return 'gm_box_pos_' . $this->getBoxPosition($p_boxName);
	}
	
	
	/**
	 * Returns error message
	 * 
	 * @return string
	 */
	public function getErrorMessage()
	{
		return $this->errorMessage;
	}
	
	
	/**
	 * Initializes the boxes configuration of the active style configuration.
	 */
	private function _initBoxes()
	{
		if($this->boxes === null)
		{
			$this->boxes = array();
			
			$styleConfig = $this->_findStyleConfig(self::$styleName);
			if($styleConfig !== null)
			{
				$styleConfigArray = $styleConfig->getJsonDataArray();
				
				foreach($styleConfigArray['settings'] as $setting)
				{
					if(!empty($setting) && $setting['type'] === 'boxes')
					{
						foreach($setting['entries'] as $entry)
						{
							$this->boxes[$entry['name']]             = array();
							$this->boxes[$entry['name']]['isActive'] = $entry['value'];
							$this->boxes[$entry['name']]['position'] = $entry['position'];
						}
					}
				}
			}
		}
	}
	
	
	/**
	 * @param $p_styleName
	 *
	 * @return null|\StyleEdit\Entities\StoredConfig
	 */
	private function _findStyleConfig($p_styleName = null)
	{
		if($p_styleName === null || !Authentication::isAuthenticated())
		{
			$styleConfig = $this->reader->findActiveStyleConfig();
		}
		else
		{
			try
			{
				$styleConfig = $this->reader->getStyleConfigByName($p_styleName);
			}
			catch(\Exception $e)
			{
				$styleConfig = $this->reader->findActiveStyleConfig();
			}
		}

		if($styleConfig === null)
		{
			try
			{
				$styleConfig = $this->reader->getBoilerplateByName('boilerplate1');
			}
			catch(\Exception $e)
			{
				$styleConfig = null;
			}
		}
		
		return $styleConfig;
	}


    /**
     * @param int $position
     * @return bool
     */
    public function getStatusByPosition(int $position): bool
    {
        $this->_initBoxes();

        if ($this->boxes) {
            foreach ($this->boxes as $box) {
                if (!empty($box['position']) && $box['position'] === $position) {
                    return $box['isActive'];
                }
            }
        }

        return false;
    }
}
